#!/usr/bin/env python

from __future__ import print_function

import carla

from srunner.autoagents.sensor_interface import CallBack
from srunner.scenariomanager.carla_data_provider import CarlaDataProvider


class AgentWrapper(object):

    """
    works as an interface between the agent, controlled vehicle and the environment
    It unifies apis multiple different kinds of agents
    """

    _agent = None
    _sensors_list = []

    def __init__(self, agent, vehicle=None, agent_config=None):
        """
        Set the autonomous agent, and connect it to the vehicle
        """
        self._agent = agent
        self._agent_id = agent.agent_id
        self._vehicle = vehicle
        self._vehicle_id = vehicle.id
        self._agent_config = agent_config
        self._agent_type = agent_config.agent_type
        self.is_alive = True
        self.is_focal = agent_config.get("is_focal", False)

    def __call__(self):
        """
        Pass the call directly to the agent
        """
        return self._agent()

    def reset(self):
        self.is_alive = True

    def setup_sensors(self, vehicle, debug_mode=False):
        """
        Create the sensors defined by the user and attach them to the ego-vehicle
        :param vehicle: ego vehicle
        :return:
        """
        bp_library = CarlaDataProvider.get_world().get_blueprint_library()
        for sensor_spec in self._agent.sensors():
            # These are the sensors spawned on the carla world
            bp = bp_library.find(str(sensor_spec['type']))
            if sensor_spec['type'].startswith('sensor.camera'):
                bp.set_attribute('image_size_x', str(sensor_spec['width']))
                bp.set_attribute('image_size_y', str(sensor_spec['height']))
                bp.set_attribute('fov', str(sensor_spec['fov']))
                sensor_location = carla.Location(x=sensor_spec['x'], y=sensor_spec['y'],
                                                 z=sensor_spec['z'])
                sensor_rotation = carla.Rotation(pitch=sensor_spec['pitch'],
                                                 roll=sensor_spec['roll'],
                                                 yaw=sensor_spec['yaw'])
            elif sensor_spec['type'].startswith('sensor.lidar'):
                bp.set_attribute('range', str(sensor_spec['range']))
                bp.set_attribute('rotation_frequency', str(sensor_spec['rotation_frequency']))
                bp.set_attribute('channels', str(sensor_spec['channels']))
                bp.set_attribute('upper_fov', str(sensor_spec['upper_fov']))
                bp.set_attribute('lower_fov', str(sensor_spec['lower_fov']))
                bp.set_attribute('points_per_second', str(sensor_spec['points_per_second']))
                sensor_location = carla.Location(x=sensor_spec['x'], y=sensor_spec['y'],
                                                 z=sensor_spec['z'] + vehicle.bounding_box.extent.z*2)
                sensor_rotation = carla.Rotation(pitch=sensor_spec['pitch'],
                                                 roll=sensor_spec['roll'],
                                                 yaw=sensor_spec['yaw'])
            elif sensor_spec['type'].startswith('sensor.other.gnss'):
                sensor_location = carla.Location(x=sensor_spec['x'], y=sensor_spec['y'],
                                                 z=sensor_spec['z'])
                sensor_rotation = carla.Rotation()

            # create sensor
            sensor_transform = carla.Transform(sensor_location, sensor_rotation)
            sensor = CarlaDataProvider.get_world().spawn_actor(bp, sensor_transform, vehicle)
            # setup callback
            sensor.listen(CallBack(sensor_spec['id'], sensor, self._agent.sensor_interface))
            self._sensors_list.append(sensor)

        # Tick once to spawn the sensors
        CarlaDataProvider.get_world().tick()

    def observe(self):
        """
        get observation from the sensors and possibly the environment caption and messages
        """
        try:
            if self._agent_config.agent_type in ['comm_agent', 'comm_only_agent',\
                                                 'control_only_agent',\
                                                 'coopernaut_receiver_agent',\
                                                 'coopernaut_sender_agent']:
                """
                Communication agent needs to collect messages
                """
                obs = self._agent.get_observation()
            else:
                """
                Sensor based agent gets data from the sensors
                """
                obs = self._agent.sensor_interface.get_data()
            return obs
        except:
            return

    def apply_control(self, action):
        """
        process the action and send it to the agent
        :action: action can be a vehicle control type,
                 or a dictionary of control and messages,
                 agents can transform the dictory to a vehicle control type
        """
        if type(action) is dict:
            """
            Some agent may have more actions space than VehicleControl
            """
            action = self._agent.run_step(action)
        elif self._agent_type in ['behavior_agent']:
            """
            Some agents does not require action input
            """
            action = self._agent.run_step()
        """
        Apply VehicleControl to vehicle, other actions are handled inside the agent
        :action: carla.VehicleControl
        """
        if action is not None:
            self._vehicle.apply_control(action)

    def cleanup(self):
        """
        Remove and destroy all sensors
        """
        for i, _ in enumerate(self._sensors_list):
            if self._sensors_list[i] is not None:
                self._sensors_list[i].stop()
                self._sensors_list[i].destroy()
                self._sensors_list[i] = None
        self._sensors_list = []
